"use client";
import { useState, useEffect, useRef } from "react";
import {
  Camera, User, Mail, Phone, Cake, Languages, ChevronRight,
  ArrowLeft, CheckCircle2, Loader2, Lock, AlertCircle, Calendar, ShieldCheck
} from "lucide-react";
import { cn } from "@/lib/utils";
import CustomCalendar from '@/components/CustomCalendar';
import { useAuth } from "@/context/AuthContext"
import InfoRow from "@/components/dashboard/InfoRow"
// --- Components ---



export default function PersonalInfoPage() {
  const [editingField, setEditingField] = useState<string | null>(null);
  const [isSaving, setIsSaving] = useState(false);
  const [showToast, setShowToast] = useState(false);
  const [toastMsg, setToastMsg] = useState("");
  const [isResending, setIsResending] = useState(false);
  const [resendCooldown, setResendCooldown] = useState(0);

  const [isVerifying, setIsVerifying] = useState(false);
  const [otp, setOtp] = useState("");
  const [isDirty, setIsDirty] = useState(false);



  const [verifyingField, setVerifyingField] = useState<string | null>(null);
  const [otpMap, setOtpMap] = useState<Record<string, string>>({});
  const [cooldownMap, setCooldownMap] = useState<Record<string, number>>({});


  // User Data State
  const { user, loading, refreshUser } = useAuth()
  const [profile, setProfile] = useState({
    firstName: user?.first_name,
    lastName: user?.last_name,
    gender: user?.gender,
    primaryPhone: user?.phone_number,
    primaryPhoneVerified: true,
    recoveryPhone: user?.recovery_phone,
    recoveryPhoneVerified: user?.recovery_phone_verified,
    primaryEmail: user?.primary_email,
    primaryEmailVerified: user?.primary_email_verified, // User needs to verify
    recoveryEmail: user?.recovery_email,
    recoveryEmailVerified: user?.recovery_email_verified,
    birthday: user?.dob
  });



  const [selectedDate, setSelectedDate] = useState(profile.birthday);
  const [isOpen, setIsOpen] = useState(false);
  const containerRef = useRef(null);



  useEffect(() => {
    const handleClickOutside = (event) => {
      if (containerRef.current && !containerRef.current.contains(event.target)) {
        setIsOpen(false);
      }
    };
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  const [editBuffer, setEditBuffer] = useState<any>(null);

  const startEdit = (field: string) => {
    setEditingField(field);
    setEditBuffer({ ...profile });


    setVerifyingField(null);
  };


  const closeEdit = () => {
    setEditingField(null);
    // setIsVerifying(false);
    setIsDirty(false);
    setOtp("");
    setToastMsg("");
    setShowToast(false);

    setVerifyingField(null);
    setOtpMap({});
  };




  useEffect(() => {
    if (showToast) {
      const timer = setTimeout(() => setShowToast(false), 4000);
      return () => clearTimeout(timer);
    }
  }, [showToast]);


  useEffect(() => {
    let interval;
    if (resendCooldown > 0) {
      interval = setInterval(() => {
        setResendCooldown((current) => current - 1);
      }, 1000);
    }
    return () => clearInterval(interval);
  }, [resendCooldown]);


  const prevValueRef = useRef("");

  const [sendOTPFor, setSendOTPFor] = useState("")




  const PURPOSE = {
    primaryEmail: "primary_email",
    recoveryPhone: "recovery_phone",
    recoveryEmail: "recovery_email"

  }

  useEffect(() => {
    if (otp.length === 6) {
      verifyOtp();
    }
  }, [otp]);

  const verifyOtp = async () => {
    setIsSaving(true);
    try {
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/recovery/otp/verify/`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({
          identifier: editBuffer[editingField],
          otp: otp,
          purpose: PURPOSE[editingField],
        }),
      });

      if (response.ok) {
        await refreshUser(); // Update global auth state
        setEditingField(null);
        setIsVerifying(false);
        setToastMsg("Identity verified and updated");
        setShowToast(true);

      } else {
        setToastMsg("Invalid code, please try again");
        setShowToast(true);
        setOtp(""); // Reset input on failure
      }
    } catch (error) {
      setToastMsg("Verification failed");
    } finally {
      setIsSaving(false);
      setOtp("");
    }
  };


  useEffect(() => {
    if (user) {
      setProfile({
        firstName: user.first_name,
        lastName: user.last_name,
        gender: user.gender,
        primaryPhone: user.phone_number,
        primaryPhoneVerified: true,
        recoveryPhone: user.recovery_phone,
        recoveryPhoneVerified: user.recovery_phone_verified,
        primaryEmail: user.primary_email,
        primaryEmailVerified: user.primary_email_verified,
        recoveryEmail: user.recovery_email,
        recoveryEmailVerified: user.recovery_email_verified,
        birthday: user.dob
      });
      setIsDirty(false)



    }
  }, [user]);


  const handleSaveStandard = async () => {
    setIsSaving(true);
    try {
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/accounts/update-profile/`, {
        method: "PATCH",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({
          first_name: editBuffer.firstName,
          last_name: editBuffer.lastName,
          dob: editBuffer.birthday,
          gender: editBuffer.gender,
        }),
      });

      if (response.ok) {
        await refreshUser();
        setToastMsg("Profile updated");
        setShowToast(true);
        setEditingField(null);
      }
    } catch (e) { console.error(e); }
    finally { setIsSaving(false); }
  };

  const handleTriggerVerification = async () => {
    setIsResending(true);

    // if ((prevValueRef.current == editBuffer[editingField])) {
    //   setIsVerifying(true)
    //   return
    // }
    // prevValueRef.current = editBuffer[editingField] || ""
    try {
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/otp/recovery/identifier/send/`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({
          identifier: editBuffer[editingField!],
          purpose: PURPOSE[editingField!]
        }),
      });

      if (response.ok) {
        setSendOTPFor(editingField);
        setIsVerifying(true);
        setResendCooldown(30);
        setToastMsg("Code sent successfully");
      } else {
        const data = await response.json();
        setToastMsg(data.error || "Failed to send code");
      }
      setShowToast(true);
    } finally { setIsResending(false); }
  };


  const [isChanged, setIsChanged] = useState(false)


  const VERIFICATION_STATUS = {
    primaryEmail: user?.primary_email_verified,
    recoveryPhone: user?.recovery_phone_verified,
    recoveryEmail: user?.recovery_email_verified
  }

  useEffect(() => {
    try {
      if (!editingField) return;

      let currentValue = user[PURPOSE[editingField]]
      let editingField = editBuffer[editingField]


      if (currentValue != editBuffer[editingField]) {
        setIsChanged(true)
      } else {
        setIsChanged(false)
      }
    } catch (e) { }

  }, [editingField ? editBuffer[editingField] : null])



  // --- Sub-Renders ---

  const renderStandardEditor = () => (
    <div className="space-y-6" >
      {editingField === "name" && (
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="relative">
            <input className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-xl px-4 py-3 outline-none"
              value={editBuffer.firstName} onChange={(e) => { setEditBuffer({ ...editBuffer, firstName: e.target.value }); setIsDirty(true); }} />
            <label className="absolute left-3 -top-2.5 px-2 bg-card text-xs text-sidebar-primary">First Name</label>
          </div>
          <div className="relative">
            <input className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-xl px-4 py-3 outline-none"
              value={editBuffer.lastName} onChange={(e) => { setEditBuffer({ ...editBuffer, lastName: e.target.value }); setIsDirty(true); }} />
            <label className="absolute left-3 -top-2.5 px-2 bg-card text-xs text-sidebar-primary">Last Name</label>
          </div>
        </div>
      )}


      {editingField === "birthday" && (
        <div className="relative"
          onClick={() => setIsOpen(!isOpen)}
        >
          <input
            type="text"
            className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-xl px-4 py-4 outline-none appearance-none"
            value={editBuffer.birthday}
            onChange={(e) => setEditBuffer({ ...editBuffer, birthday: e.target.value })}
          />
          <Calendar className="absolute right-4 top-4 text-muted-foreground pointer-events-none" size={20} />

          <label className="absolute left-3 -top-2.5 px-2 bg-card text-xs text-sidebar-primary">Select Birthday</label>
        </div>
      )}

      {isOpen && (
        <div className="fixed w-full h-full top-0 left-0 z-50 flex justify-center align-middle animate-in fade-in zoom-in-95 duration-500" style={{
          alignItems: 'center',
          backdropFilter: 'blur(10px)'
        }}
          onClick={() => setIsOpen(false)}

        >
          <div
            onClick={(e) => e.stopPropagation()}
            className="relative"
          >
            <CustomCalendar
              selectedDate={editBuffer.birthday} // "2002-08-10"
              onSelect={(newDate) => { setEditBuffer({ ...editBuffer, birthday: newDate }); setIsDirty(true); setIsOpen(false) }}

            /></div>
        </div>
      )}


      {/* GENDER SELECTION */}
      {editingField === "gender" && (
        <div className="space-y-4">
          <label className="text-[11px] font-bold text-muted-foreground uppercase tracking-wider ml-1">
            Select Gender
    </label>
          <div className="grid grid-cols-1 sm:grid-cols-3 gap-3">
            {['Male', 'Female', 'Other'].map((option) => (
              <button
                key={option}
                onClick={() => {
                  setEditBuffer({ ...editBuffer, gender: option });
                  setIsDirty(true);
                }}
                className={cn(
                  "flex items-center justify-between px-4 py-4 rounded-xl border-2 transition-all text-sm font-medium",
                  editBuffer.gender === option
                    ? "border-sidebar-primary bg-sidebar-primary/5 text-sidebar-primary"
                    : "border-muted hover:border-muted-foreground/30 text-muted-foreground"
                )}
              >
                {option}
                <div className={cn(
                  "w-4 h-4 rounded-full border-2 flex items-center justify-center",
                  editBuffer.gender === option ? "border-sidebar-primary" : "border-muted"
                )}>
                  {editBuffer.gender === option && (
                    <div className="w-2 h-2 rounded-full bg-sidebar-primary animate-in zoom-in-50" />
                  )}
                </div>
              </button>
            ))}
          </div>
          <p className="text-xs text-muted-foreground px-1 mt-2">
            This helps us personalize your experience across the platform.
    </p>
        </div>
      )}


      <div className="flex justify-end gap-3 pt-4 border-t">
        <button onClick={closeEdit} className="px-5 py-2 text-sm font-semibold text-sidebar-primary">Cancel</button>
        <button onClick={handleSaveStandard} disabled={!isDirty || isSaving} className="bg-sidebar-primary text-white px-8 py-2 rounded-lg text-sm font-semibold disabled:opacity-50">
          {isSaving ? <Loader2 className="animate-spin" size={16} /> : "Save"}
        </button>
      </div>
    </div>
  );


  // 1. Inside your PersonalInfoPage component (top level)
  const initialValueRef = useRef<string | null>(null);

  // 2. Update the ref whenever the user starts editing a new field
  useEffect(() => {
    if (editingField) {
      // Store the value currently in the user object as the "truth"
      console.log(user[PURPOSE[editingField]])
      initialValueRef.current = user[PURPOSE[editingField]] || "";
    } else {
      initialValueRef.current = null;
    }
  }, [editingField ? editBuffer[editingField] : null, user]);




  const renderVerificationEditor_old = () => {

    const isFieldVerified = user[`${editingField?.replace(/([A-Z])/g, '_$1').toLowerCase()}_verified`];


    const dialogConfig = !isChanged ? {
      title: "Confirm Information Change",
      description: `You are changing your ${editingField}. For security, we need to verify this via your Primary Phone.`,
      buttonText: "Verify & Update",
      colorClass: "bg-indigo-500/5 border-indigo-500/30 text-indigo-600",
      icon: <ShieldCheck size={18} />
    } : {
        title: "Verification Required",
        description: `Please verify your ${editingField} to ensure your account remains secure.`,
        buttonText: "Send Verification Code",
        colorClass: "bg-amber-500/5 border-amber-500/30 text-amber-600",
        icon: <AlertCircle size={18} />
      };

    return (
      <div className="space-y-6" ref={containerRef}>
        {/* Input Section */}
        <div className="relative">
          <input
            type={editingField?.includes("Email") ? "email" : "tel"}
            disabled={isVerifying}
            className={cn(
              "w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-xl px-4 py-4 outline-none transition-all",
              isVerifying && "opacity-50 bg-muted/10"
            )}
            value={editBuffer[editingField!]}
            onChange={(e) => {
              setEditBuffer({ ...editBuffer, [editingField!]: e.target.value });
              setIsDirty(true);
              setIsVerifying(false);

            }}
          />
          <label className="absolute left-3 -top-2.5 px-2 bg-card text-xs text-sidebar-primary uppercase font-bold">
            {editingField?.replace(/([A-Z])/g, ' $1')}
          </label>
          <label className="absolute left-3 -top-2.5 px-2 bg-card text-xs text-sidebar-primary uppercase flex items-center gap-1">
            {editingField?.replace(/([A-Z])/g, ' $1')}
            {isFieldVerified && !isDirty && (<span className="flex gap-1 text-green-500" ><CheckCircle2 size={12} className="text-green-500" /> Verified</span>)}
          </label>
        </div>

        {/* DYNAMIC INITIAL DIALOG */}
        {((!isVerifying && editBuffer[editingField!]) || sendOTPFor != editingField) && (!isFieldVerified || isDirty) && editBuffer[editingField!] && (
          <div className={cn("p-5 rounded-2xl border flex flex-col gap-4 animate-in fade-in zoom-in-95 duration-300", dialogConfig.colorClass)}>
            <div className="flex items-start gap-3">
              <div className="mt-0.5">{dialogConfig.icon}</div>
              <div className="space-y-1">
                <h4 className="font-bold text-sm leading-none">{dialogConfig.title}</h4>
                <p className="text-xs opacity-80 leading-relaxed">
                  {dialogConfig.description}
                </p>
              </div>
            </div>

            <button
              onClick={handleTriggerVerification}
              disabled={!editBuffer[editingField!]}
              // className="w-full py-3 rounded-xl bg-current font-bold text-white transition-transform active:scale-95 flex items-center justify-center gap-2"
              className="text-xs font-bold bg-sidebar-primary text-white px-4 py-2 rounded-lg disabled:opacity-50"
            >
              <span className="filter brightness-0 invert">{dialogConfig.buttonText}</span>
            </button>
          </div>
        )}

        {/* OTP INPUT SECTION (Shows after clicking button) */}
        {isVerifying && sendOTPFor == editingField && (
          <div className="p-6 bg-sidebar-primary/5 rounded-2xl border border-sidebar-primary/20 space-y-4 text-center animate-in slide-in-from-bottom-2">
            <div className="space-y-1">
              <p className="text-sm font-bold uppercase tracking-wider opacity-60">Enter Code</p>
              <p className="text-xs">Sent to {isChanged ? "Primary Phone" : editBuffer[editingField!]}</p>
            </div>

            <input
              maxLength={6}
              autoFocus
              className="w-full max-w-[200px] text-center bg-transparent border-b-2 border-sidebar-primary text-3xl tracking-[0.4em] font-black outline-none py-2"
              value={otp}
              onChange={(e) => setOtp(e.target.value.replace(/\D/g, ""))}
            />

            <button
              onClick={handleTriggerVerification}
              disabled={resendCooldown > 0}
              className="text-xs font-bold text-sidebar-primary block mx-auto hover:underline"
            >
              {resendCooldown > 0 ? `Resend in ${resendCooldown}s` : "Didn't get a code? Resend"}
            </button>
          </div>
        )}
      </div>
    );
  };





  const renderVerificationEditor = () => {
    

    if (!editingField) return null;

    // 1. Data Context
    const dbFieldName = editingField.replace(/([A-Z])/g, '_$1').toLowerCase();
    const isFieldVerified = user[`${dbFieldName}_verified`];

    const originalValue = user[PURPOSE[editingField]] //initialValueRef.current ?? "";
    const currentValue = editBuffer[editingField] || "";

    // 2. State Logic
    const hasValue = currentValue.trim().length > 0;
    const hasChanged = originalValue !== "" && currentValue !== originalValue;
    // If original was empty but now has value, we treat it as an "Update" (Phone Verification)
    const isUpdate = hasChanged || (originalValue === "" && hasValue);
    const isUnverifiedOriginal = hasValue && !hasChanged && !isFieldVerified;

    // 3. Logic-Driven Configuration
    let dialogConfig = null;

    if (isUpdate) {
      dialogConfig = {
        title: "Security Verification",
        description: `Updating your information requires authorization via your primary phone: ${user.phone_number}`,
        buttonText: "Verify via Phone",
        colorClass: "bg-indigo-500/5 border-indigo-500/30 text-indigo-600",
        icon: <ShieldCheck size={18} />
      };
    } else if (isUnverifiedOriginal) {
      dialogConfig = {
        title: "Verify This Field",
        description: `This ${editingField} hasn't been verified yet. Confirm ownership to secure your account.`,
        buttonText: "Send Code to Field",
        colorClass: "bg-amber-500/5 border-amber-500/30 text-amber-600",
        icon: <AlertCircle size={18} />
      };
    }

    // 4. Final Visibility Check
    // Show only if we HAVE a config (Update or Unverified) AND we aren't already in the OTP screen
    const shouldShowPrompt = !isVerifying && !!dialogConfig && hasValue;

    return (
      <div className="space-y-6" ref={containerRef}>
        {/* Input Section */}
        <div className="relative">
          <input
            type={editingField?.includes("Email") ? "email" : "tel"}
            disabled={isVerifying}
            className={cn(
              "w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-xl px-4 py-4 outline-none transition-all",
              isVerifying && "opacity-50 bg-muted/10"
            )}
            value={currentValue || ""}
            onChange={(e) => {
              setEditBuffer({ ...editBuffer, [editingField!]: e.target.value });
              setIsDirty(true);
              setIsVerifying(false);
            }}
          />
          <label className="absolute left-3 -top-2.5 px-2 bg-card text-xs text-sidebar-primary uppercase flex items-center gap-1 font-bold">
            {editingField?.replace(/([A-Z])/g, ' $1')}
            {isFieldVerified && !isDirty && (
              <span className="flex items-center gap-1 text-green-600 ml-1">
                <CheckCircle2 size={12} /> Verified
              </span>
            )}
          </label>
        </div>

        {/* DYNAMIC PROMPT: Show if not verifying, field has value, and is either dirty or unverified */}
        {/* !isVerifying && currentValue && (!isFieldVerified || isDirty) && */}
        {shouldShowPrompt && (
          <div className={cn("p-5 rounded-2xl border flex flex-col gap-4 animate-in fade-in zoom-in-95 duration-300", dialogConfig.colorClass)}>
            <div className="flex items-start gap-3">
              <div className="mt-0.5">{dialogConfig.icon}</div>
              <div className="space-y-1">
                <h4 className="font-bold text-sm leading-none">{dialogConfig.title}</h4>
                <p className="text-xs opacity-80 leading-relaxed">
                  {dialogConfig.description}
                </p>
              </div>
            </div>

            <button
              onClick={handleTriggerVerification}
              className={cn("text-xs font-bold bg-sidebar-primary text-white px-4 py-2 rounded-lg hover:opacity-90 transition-opacity", dialogConfig.colorClass)}
            >
              {dialogConfig.buttonText}
            </button>
          </div>
        )}

        {/* OTP INPUT SECTION */}
        {isVerifying && sendOTPFor === editingField && (
          <div className="p-6 bg-sidebar-primary/5 rounded-2xl border border-sidebar-primary/20 space-y-4 text-center animate-in slide-in-from-bottom-2">
            <div className="space-y-1">
              <p className="text-sm font-bold uppercase tracking-wider opacity-60">Enter Code</p>
              <p className="text-xs font-medium">
                Sent to: <span className="text-sidebar-primary">
                  {isUpdate ? user?.phone_number : currentValue}
                </span>
              </p>
            </div>

            <input
              maxLength={6}
              autoFocus
              className="w-full max-w-[200px] text-center bg-transparent border-b-2 border-sidebar-primary text-3xl tracking-[0.4em] font-black outline-none py-2"
              value={otp}
              onChange={(e) => setOtp(e.target.value.replace(/\D/g, ""))}
            />

            <button
              onClick={handleTriggerVerification}
              disabled={resendCooldown > 0}
              className="text-xs font-bold text-sidebar-primary block mx-auto hover:underline disabled:opacity-50"
            >
              {resendCooldown > 0 ? `Resend in ${resendCooldown}s` : "Didn't get a code? Resend"}
            </button>
          </div>
        )}
      </div>
    );
  };

  
  // --- Main Render ---

  if (loading) return <div className="h-screen flex items-center justify-center"><Loader2 className="animate-spin" /></div>;

  if (editingField) {
    const isVerificationType = ["primaryEmail", "recoveryEmail", "recoveryPhone"].includes(editingField);
    return (
      <div className="max-w-3xl mx-auto pt-8 pb-20 px-4">
        <button onClick={closeEdit} className="flex items-center gap-2 text-muted-foreground mb-8">
          <ArrowLeft size={20} /> <span className="text-sm">Back</span>
        </button>
        <h1 className="text-3xl font-normal mb-6 capitalize">{editingField.replace(/([A-Z])/g, ' $1')}</h1>
        <div className="bg-card border rounded-2xl p-6 shadow-sm">
          {isVerificationType ? renderVerificationEditor() : renderStandardEditor()}
        </div>

        {(showToast) && (
          <div className="fixed z-[51] bottom-8 left-8 bg-[#323232] text-white px-6 py-3 rounded-lg shadow-2xl flex items-center gap-3 animate-in slide-in-from-left-10">
            <CheckCircle2 size={18} className="text-green-400" />
            <span className="text-sm">{toastMsg}</span>
          </div>
        )}
      </div>
    );
  }

  return (
    <div className="max-w-3xl mx-auto pt-8 pb-20 px-4 relative">
      <div className="mb-8">
        <h1 className="text-3xl font-normal mb-2">Personal info</h1>
        <p className="text-sm text-muted-foreground">Manage your identity and contact methods across Indzs.</p>
      </div>


      <div className="bg-background border border-border rounded-2xl overflow-hidden shadow-sm">

        <div className="flex items-center justify-between p-6 hover:bg-accent/30 cursor-pointer transition-colors border-b border-border group">
          <div className="flex items-center gap-6">
            <Camera size={20} className="text-muted-foreground" />
            <span className="text-base font-medium">Profile picture</span>
          </div>
          <div className="w-16 h-16 rounded-full bg-sidebar-primary/20 flex items-center justify-center text-2xl text-sidebar-primary relative overflow-hidden group">
            {profile.firstName.toUpperCase().charAt(0)}
            <div className="absolute inset-0 bg-black/40 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center">
              <Camera size={20} className="text-white" />
            </div>
          </div>
        </div>
        <InfoRow icon={User} label="Name" value={`${profile.firstName} ${profile.lastName}`} onClick={() => startEdit("name")} />
        <InfoRow
          icon={Mail}
          label="Primary Email"
          value={profile.primaryEmail}
          needsVerification={!profile.primaryEmailVerified}
          onClick={() => startEdit("primaryEmail")}
        />
        <InfoRow
          icon={Mail}
          label="Recovery Email"
          value={profile.recoveryEmail || "Not added"}
          needsVerification={profile.recoveryEmail && !profile.recoveryEmailVerified}
          onClick={() => startEdit("recoveryEmail")}
        />
        <InfoRow icon={Phone} label="Primary Phone" value={profile.primaryPhone} isLocked />
        <InfoRow
          icon={Phone}
          label="Recovery Phone"
          value={profile.recoveryPhone || "None added"}
          needsVerification={profile.recoveryPhone && !profile.recoveryPhoneVerified}
          onClick={() => startEdit("recoveryPhone")}
        />
        <InfoRow icon={Cake} label="Birthday" value={profile.birthday ? new Date(profile.birthday).toLocaleDateString('en-GB', { day: 'numeric', month: 'long', year: 'numeric' }): "Not Added"} onClick={() => startEdit("birthday")} />

        <InfoRow
          icon={User} // or a custom 'VenetianMask' / 'Users' icon
          label="Gender"
          value={profile.gender || "Not specified"}
          onClick={() => startEdit("gender")}
        />

      </div>
      {(showToast) && (
        <div className="fixed z-[51] bottom-8 left-8 bg-[#323232] text-white px-6 py-3 rounded-lg shadow-2xl flex items-center gap-3 animate-in slide-in-from-left-10">
          <CheckCircle2 size={18} className="text-green-400" />
          <span className="text-sm">{toastMsg}</span>
        </div>
      )}
    </div>
  );
}




